<?php
// Configuración detallada de errores
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';
if (!$conn) {
    die("Error de conexión: " . $conn->connect_error);
}
$conn->set_charset("utf8mb4");

// Obtener listados para filtros
$sql_postgrados = "SELECT * FROM postgrados ORDER BY nombre";
$result_postgrados = $conn->query($sql_postgrados);

$sql_promociones = "SELECT * FROM promociones ORDER BY nombre";
$result_promociones = $conn->query($sql_promociones);

// Obtener parámetros de filtro
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$promocion_id = isset($_GET['promocion_id']) ? intval($_GET['promocion_id']) : 0;
$cedula_filtro = isset($_GET['cedula']) ? trim($_GET['cedula']) : '';

// Obtener graduados que no cumplen requisitos - CONSULTA CORREGIDA
$graduados_irregulares = [];
$sql_irregulares = "
    SELECT 
        e.id,
        e.nombres,
        e.apellidos,
        e.cedula,
        p.nombre AS postgrado_nombre,
        p.id AS postgrado_id,
        pr.nombre AS promocion_nombre,
        ep.fecha_graduacion,
        COUNT(d.id) AS total_materias,
        SUM(CASE WHEN i.nota_definitiva IS NULL THEN 1 ELSE 0 END) AS materias_sin_nota,
        SUM(CASE 
            -- Para materias de puntaje: reprobado si < 10
            WHEN d.tipo = 'puntaje' AND CAST(i.nota_definitiva AS DECIMAL(5,2)) < 10 THEN 1 
            -- Para materias Aprobado/Reprobado: reprobado si NO es 'APROBADO'
            WHEN d.tipo = 'Aprobado_Reprobado' AND UPPER(TRIM(i.nota_definitiva)) != 'APROBADO' THEN 1 
            ELSE 0 
        END) AS materias_reprobadas
    FROM estudiantes e
    JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
    JOIN postgrados p ON ep.postgrado_id = p.id
    LEFT JOIN promociones pr ON ep.promocion_id = pr.id
    JOIN denominaciones d ON p.id = d.postgrado_id
    LEFT JOIN inscripciones i ON d.id = i.denominacion_id AND i.estudiante_id = e.id
    WHERE ep.estado = 'graduado'
    GROUP BY e.id, p.id
    HAVING materias_sin_nota > 0 OR materias_reprobadas > 0
";

$conditions = [];
$params = [];
$types = "";

if ($postgrado_id > 0) {
    $conditions[] = "p.id = ?";
    $params[] = $postgrado_id;
    $types .= "i";
}

if ($promocion_id > 0) {
    $conditions[] = "ep.promocion_id = ?";
    $params[] = $promocion_id;
    $types .= "i";
}

if (!empty($cedula_filtro)) {
    $conditions[] = "e.cedula LIKE ?";
    $params[] = "%$cedula_filtro%";
    $types .= "s";
}

if (!empty($conditions)) {
    $sql_irregulares .= " AND " . implode(" AND ", $conditions);
}

$sql_irregulares .= " ORDER BY p.nombre, e.apellidos, e.nombres";

$stmt = $conn->prepare($sql_irregulares);
if ($stmt) {
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    if ($stmt->execute()) {
        $graduados_irregulares = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    } else {
        $_SESSION['error'] = "Error al obtener graduados irregulares: " . $stmt->error;
    }
    $stmt->close();
} else {
    $_SESSION['error'] = "Error en la consulta de graduados irregulares: " . $conn->error;
}

// Función para obtener el detalle de materias problemáticas de un estudiante - CORREGIDA
function obtenerMateriasProblema($conn, $estudiante_id, $postgrado_id) {
    $sql_materias = "
        SELECT 
            d.nombre,
            d.tipo,
            i.nota_definitiva,
            CASE 
                WHEN i.nota_definitiva IS NULL THEN 'sin_nota'
                -- Para materias de puntaje: reprobado si < 10
                WHEN d.tipo = 'puntaje' AND CAST(i.nota_definitiva AS DECIMAL(5,2)) < 10 THEN 'reprobada_puntaje'
                -- Para materias Aprobado/Reprobado: reprobado si NO es 'APROBADO'
                WHEN d.tipo = 'Aprobado_Reprobado' AND UPPER(TRIM(i.nota_definitiva)) != 'APROBADO' THEN 'reprobada_cualitativa'
                ELSE 'aprobada'
            END as estado
        FROM denominaciones d
        LEFT JOIN inscripciones i ON d.id = i.denominacion_id AND i.estudiante_id = ?
        WHERE d.postgrado_id = ?
        ORDER BY d.nombre
    ";
    
    $stmt = $conn->prepare($sql_materias);
    $materias_problema = [];
    
    if ($stmt) {
        $stmt->bind_param("ii", $estudiante_id, $postgrado_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($materia = $result->fetch_assoc()) {
            if ($materia['estado'] != 'aprobada') {
                $materias_problema[] = $materia;
            }
        }
        $stmt->close();
    }
    
    return $materias_problema;
}

$page_title = "Graduados Irregulares - Auditoría";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
        <a href="gestion_graduacion.php" class="btn btn-primary ms-2">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-mortarboard" viewBox="0 0 16 16">
                <path d="M8.211 2.047a.5.5 0 0 0-.422 0l-7.5 3.5a.5.5 0 0 0 .025.917l7.5 3a.5.5 0 0 0 .372 0L14 7.14V13a1 1 0 0 0-1 1v2h3v-2a1 1 0 0 0-1-1V6.739l.686-.275a.5.5 0 0 0 .025-.917l-7.5-3.5ZM8 8.46 1.758 5.965 8 3.052l6.242 2.913L8 8.46Z"/>
                <path d="M4.176 9.032a.5.5 0 0 0-.656.327l-.5 1.7a.5.5 0 0 0 .294.605l4.5 1.8a.5.5 0 0 0 .372 0l4.5-1.8a.5.5 0 0 0 .294-.605l-.5-1.7a.5.5 0 0 0-.656-.327L8 10.466 4.176 9.032Zm-.068 1.873.22-.748 3.496 1.311a.5.5 0 0 0 .352 0l3.496-1.311.22.748L8 12.46l-3.892-1.556Z"/>
            </svg>
            Gestión de Graduación
        </a>
    </div>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <!-- Filtros -->
    <div class="card mb-4">
        <div class="card-header bg-warning text-dark">
            <h3 class="mb-0">Filtros de Auditoría</h3>
        </div>
        <div class="card-body">
            <form method="get">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label for="postgrado_id" class="form-label">Postgrado:</label>
                        <select name="postgrado_id" id="postgrado_id" class="form-select">
                            <option value="0">Todos</option>
                            <?php 
                            $result_postgrados->data_seek(0);
                            while($p = $result_postgrados->fetch_assoc()): ?>
                                <option value="<?= $p['id'] ?>" <?= $postgrado_id == $p['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($p['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="promocion_id" class="form-label">Promoción:</label>
                        <select name="promocion_id" id="promocion_id" class="form-select">
                            <option value="0">Todas</option>
                            <?php 
                            $result_promociones->data_seek(0);
                            while($prom = $result_promociones->fetch_assoc()): ?>
                                <option value="<?= $prom['id'] ?>" <?= $promocion_id == $prom['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($prom['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    
                    <div class="col-md-4">
                        <label for="cedula" class="form-label">Cédula:</label>
                        <input type="text" name="cedula" id="cedula" class="form-control" placeholder="Buscar por cédula" value="<?= htmlspecialchars($cedula_filtro) ?>">
                    </div>
                    
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-warning w-100">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-search" viewBox="0 0 16 16">
                                <path d="M11.742 10.344a6.5 6.5 0 1 0-1.397 1.398h-.001c.03.04.062.078.098.115l3.85 3.85a1 1 0 0 0 1.415-1.414l-3.85-3.85a1.007 1.007 0 0 0-.115-.1zM12 6.5a5.5 5.5 0 1 1-11 0 5.5 5.5 0 0 1 11 0z"/>
                            </svg>
                            Buscar Irregularidades
                        </button>
                    </div>
                    
                    <div class="col-md-3 d-flex align-items-end">
                        <a href="auditoria_graduados.php" class="btn btn-outline-secondary w-100">Limpiar Filtros</a>
                    </div>
                    
                    <div class="col-md-6 d-flex align-items-end">
                        <div class="alert alert-info w-100 mb-0">
                            <small>
                                <strong>Total encontrado:</strong> <?= count($graduados_irregulares) ?> graduado(s) irregular(es)
                            </small>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Graduados Irregulares -->
    <div class="card">
        <div class="card-header bg-danger text-white">
            <h3 class="mb-0">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi bi-exclamation-triangle" viewBox="0 0 16 16">
                    <path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.146.146 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.163.163 0 0 1-.054.06.116.116 0 0 1-.066.017H1.146a.115.115 0 0 1-.066-.017.163.163 0 0 1-.054-.06.176.176 0 0 1 .002-.183L7.884 2.073a.147.147 0 0 1 .054-.057zm1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767L8.982 1.566z"/>
                    <path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0zM7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0L7.1 5.995z"/>
                </svg>
                Graduados que no Cumplen Requisitos
            </h3>
        </div>
        <div class="card-body">
            <?php if (!empty($graduados_irregulares)): ?>
                <div class="alert alert-warning">
                    <strong>Advertencia:</strong> Se encontraron <?= count($graduados_irregulares) ?> graduados que no cumplen con los requisitos mínimos de graduación.
                </div>

                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Nombre</th>
                                <th>Apellidos</th>
                                <th>Cédula</th>
                                <th>Postgrado</th>
                                <th>Promoción</th>
                                <th>Fecha Graduación</th>
                                <th>Problemas Detectados</th>
                                <th>Detalle</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($graduados_irregulares as $graduado): ?>
                                <?php 
                                $materias_problema = obtenerMateriasProblema($conn, $graduado['id'], $graduado['postgrado_id']);
                                $problemas = [];
                                if ($graduado['materias_sin_nota'] > 0) {
                                    $problemas[] = $graduado['materias_sin_nota'] . " materia(s) sin nota";
                                }
                                if ($graduado['materias_reprobadas'] > 0) {
                                    $problemas[] = $graduado['materias_reprobadas'] . " materia(s) reprobada(s)";
                                }
                                ?>
                                <tr class="table-warning">
                                    <td><?= $graduado['id'] ?></td>
                                    <td><?= htmlspecialchars($graduado['nombres']) ?></td>
                                    <td><?= htmlspecialchars($graduado['apellidos']) ?></td>
                                    <td><?= htmlspecialchars($graduado['cedula']) ?></td>
                                    <td><?= htmlspecialchars($graduado['postgrado_nombre']) ?></td>
                                    <td><?= htmlspecialchars($graduado['promocion_nombre'] ?? 'Sin promoción') ?></td>
                                    <td><?= htmlspecialchars($graduado['fecha_graduacion']) ?></td>
                                    <td>
                                        <span class="badge bg-danger"><?= implode(', ', $problemas) ?></span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-info" data-bs-toggle="modal" data-bs-target="#modalDetalle<?= $graduado['id'] ?>">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-eye" viewBox="0 0 16 16">
                                                <path d="M16 8s-3-5.5-8-5.5S0 8 0 8s3 5.5 8 5.5S16 8 16 8zM1.173 8a13.133 13.133 0 0 1 1.66-2.043C4.12 4.668 5.88 3.5 8 3.5c2.12 0 3.879 1.168 5.168 2.457A13.133 13.133 0 0 1 14.828 8c-.058.087-.122.183-.195.288-.335.48-.83 1.12-1.465 1.755C11.879 11.332 10.119 12.5 8 12.5c-2.12 0-3.879-1.168-5.168-2.457A13.134 13.134 0 0 1 1.172 8z"/>
                                                <path d="M8 5.5a2.5 2.5 0 1 0 0 5 2.5 2.5 0 0 0 0-5zM4.5 8a3.5 3.5 0 1 1 7 0 3.5 3.5 0 0 1-7 0z"/>
                                            </svg>
                                            Ver Detalle
                                        </button>
                                    </td>
                                </tr>

                                <!-- Modal de Detalle -->
                                <div class="modal fade" id="modalDetalle<?= $graduado['id'] ?>" tabindex="-1" aria-labelledby="modalDetalleLabel<?= $graduado['id'] ?>" aria-hidden="true">
                                    <div class="modal-dialog modal-lg">
                                        <div class="modal-content">
                                            <div class="modal-header bg-warning">
                                                <h5 class="modal-title" id="modalDetalleLabel<?= $graduado['id'] ?>">
                                                    Detalle de Irregularidades - <?= htmlspecialchars($graduado['nombres'] . ' ' . $graduado['apellidos']) ?>
                                                </h5>
                                                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                            </div>
                                            <div class="modal-body">
                                                <div class="row mb-3">
                                                    <div class="col-md-6">
                                                        <strong>Cédula:</strong> <?= htmlspecialchars($graduado['cedula']) ?>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <strong>Postgrado:</strong> <?= htmlspecialchars($graduado['postgrado_nombre']) ?>
                                                    </div>
                                                </div>
                                                <div class="row mb-3">
                                                    <div class="col-md-6">
                                                        <strong>Promoción:</strong> <?= htmlspecialchars($graduado['promocion_nombre'] ?? 'Sin promoción') ?>
                                                    </div>
                                                    <div class="col-md-6">
                                                        <strong>Fecha Graduación:</strong> <?= htmlspecialchars($graduado['fecha_graduacion']) ?>
                                                    </div>
                                                </div>

                                                <h6 class="mt-4">Materias con Problemas:</h6>
                                                <div class="table-responsive">
                                                    <table class="table table-sm table-bordered">
                                                        <thead class="table-light">
                                                            <tr>
                                                                <th>Materia</th>
                                                                <th>Tipo</th>
                                                                <th>Nota</th>
                                                                <th>Estado</th>
                                                                <th>Problema</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($materias_problema as $materia): ?>
                                                                <tr class="<?= $materia['estado'] == 'sin_nota' ? 'table-warning' : 'table-danger' ?>">
                                                                    <td><?= htmlspecialchars($materia['nombre']) ?></td>
                                                                    <td>
                                                                        <span class="badge bg-secondary">
                                                                            <?= $materia['tipo'] == 'puntaje' ? 'Puntaje' : 'Aprobado/Reprobado' ?>
                                                                        </span>
                                                                    </td>
                                                                    <td>
                                                                        <strong><?= htmlspecialchars($materia['nota_definitiva'] ?? 'Sin nota') ?></strong>
                                                                    </td>
                                                                    <td>
                                                                        <?php if ($materia['estado'] == 'sin_nota'): ?>
                                                                            <span class="badge bg-warning">Sin Nota</span>
                                                                        <?php elseif ($materia['estado'] == 'reprobada_puntaje'): ?>
                                                                            <span class="badge bg-danger">Reprobada</span>
                                                                        <?php elseif ($materia['estado'] == 'reprobada_cualitativa'): ?>
                                                                            <span class="badge bg-danger">Reprobada</span>
                                                                        <?php endif; ?>
                                                                    </td>
                                                                    <td>
                                                                        <?php if ($materia['estado'] == 'sin_nota'): ?>
                                                                            No tiene nota registrada
                                                                        <?php elseif ($materia['estado'] == 'reprobada_puntaje'): ?>
                                                                            Puntaje menor a 10 (<?= htmlspecialchars($materia['nota_definitiva']) ?>)
                                                                        <?php elseif ($materia['estado'] == 'reprobada_cualitativa'): ?>
                                                                            Calificación no es "APROBADO" (<?= htmlspecialchars($materia['nota_definitiva']) ?>)
                                                                        <?php endif; ?>
                                                                    </td>
                                                                </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="alert alert-success">
                    <?php if ($postgrado_id > 0 || $promocion_id > 0 || !empty($cedula_filtro)): ?>
                        No se encontraron graduados irregulares con los filtros aplicados.
                    <?php else: ?>
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" fill="currentColor" class="bi bi-check-circle" viewBox="0 0 16 16">
                            <path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/>
                            <path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/>
                        </svg>
                        ¡Excelente! No se encontraron graduados irregulares. Todos los graduados cumplen con los requisitos.
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php
include '../includes/footer.php';
$conn->close();
?>