<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");
// Procesar asignación de postgrados
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['asignar_postgrados'])) {
        $coordinador_id = intval($_POST['coordinador_id']);
        $postgrados = isset($_POST['postgrados']) ? $_POST['postgrados'] : [];
        
        $conn->begin_transaction();
        try {
            // Eliminar asignaciones anteriores
            $delete_sql = "DELETE FROM coordinador_postgrado WHERE coordinador_id = ?";
            $stmt = $conn->prepare($delete_sql);
            $stmt->bind_param("i", $coordinador_id);
            $stmt->execute();
            
            // Insertar nuevas asignaciones
            $insert_sql = "INSERT INTO coordinador_postgrado (coordinador_id, postgrado_id) VALUES (?, ?)";
            $stmt = $conn->prepare($insert_sql);
            
            foreach ($postgrados as $postgrado_id) {
                $postgrado_id = intval($postgrado_id);
                $stmt->bind_param("ii", $coordinador_id, $postgrado_id);
                $stmt->execute();
            }
            
            $conn->commit();
            $_SESSION['success'] = "Postgrados asignados correctamente al coordinador.";
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error'] = "Error al asignar postgrados: " . $e->getMessage();
        }
        header("Location: asignar_postgrados.php");
        exit();
    }
}

// Procesar eliminación de asignación
if (isset($_GET['eliminar'])) {
    $asignacion_id = intval($_GET['eliminar']);
    
    $sql = "DELETE FROM coordinador_postgrado WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $asignacion_id);
    
    if ($stmt->execute()) {
        $_SESSION['success'] = "Asignación eliminada correctamente.";
    } else {
        $_SESSION['error'] = "Error al eliminar la asignación.";
    }
    
    header("Location: asignar_postgrados.php");
    exit();
}

// Obtener lista de coordinadores
$sql_coordinadores = "SELECT id, username FROM usuarios WHERE rol = 'coordinador2' ORDER BY username";
$coordinadores = $conn->query($sql_coordinadores);

// Obtener lista de postgrados
$sql_postgrados = "SELECT id, nombre FROM postgrados ORDER BY nombre";
$postgrados = $conn->query($sql_postgrados);

// Obtener todas las asignaciones existentes
$sql_asignaciones = "SELECT cp.id, u.username AS coordinador, p.nombre AS postgrado 
                    FROM coordinador_postgrado cp
                    JOIN usuarios u ON cp.coordinador_id = u.id
                    JOIN postgrados p ON cp.postgrado_id = p.id
                    ORDER BY u.username, p.nombre";
$asignaciones = $conn->query($sql_asignaciones);

$page_title = "Gestión de Asignaciones - Coordinadores y Postgrados";
include '../includes/header.php';
?>

<div class="container">
    <!-- Botón para volver al panel -->
    <div class="mb-4">
        <a href="index.php" class="btn btn-secondary">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Volver al Panel
        </a>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success"><?= $_SESSION['success'] ?></div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger"><?= $_SESSION['error'] ?></div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <div class="card">
        <div class="card-header bg-primary text-white">
            <h3 class="mb-0">Asignar Postgrados a Coordinador</h3>
        </div>
        <div class="card-body">
            <form method="post" id="asignacionForm">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label for="coordinador_id" class="form-label">Seleccionar Coordinador:</label>
                        <select name="coordinador_id" id="coordinador_id" class="form-select" required>
                            <option value="">-- Seleccione un coordinador --</option>
                            <?php while ($coordinador = $coordinadores->fetch_assoc()): ?>
                                <option value="<?= $coordinador['id'] ?>"><?= htmlspecialchars($coordinador['username']) ?></option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                </div>

                <div class="row g-3 mt-2">
                    <div class="col-12">
                        <label class="form-label">Postgrados Disponibles:</label>
                        <div class="postgrados-container">
                            <?php 
                            // Reset pointer para poder iterar nuevamente
                            $postgrados->data_seek(0);
                            while ($postgrado = $postgrados->fetch_assoc()): ?>
                                <div class="postgrado-item">
                                    <input type="checkbox" name="postgrados[]" value="<?= $postgrado['id'] ?>" id="postgrado_<?= $postgrado['id'] ?>">
                                    <label for="postgrado_<?= $postgrado['id'] ?>" class="form-check-label"><?= htmlspecialchars($postgrado['nombre']) ?></label>
                                </div>
                            <?php endwhile; ?>
                        </div>
                    </div>
                </div>

                <div class="row g-3 mt-3">
                    <div class="col-12">
                        <button type="submit" name="asignar_postgrados" class="btn btn-primary">
                            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-save" viewBox="0 0 16 16">
                                <path d="M2 1a1 1 0 0 0-1 1v12a1 1 0 0 0 1 1h12a1 1 0 0 0 1-1V2a1 1 0 0 0-1-1H9.5a1 1 0 0 0-1 1v7.293l2.646-2.647a.5.5 0 0 1 .708.708l-3.5 3.5a.5.5 0 0 1-.708 0l-3.5-3.5a.5.5 0 1 1 .708-.708L7.5 9.293V2a2 2 0 0 1 2-2H14a2 2 0 0 1 2 2v12a2 2 0 0 1-2 2H2a2 2 0 0 1-2-2V2a2 2 0 0 1 2-2h2.5a.5.5 0 0 1 0 1H2z"/>
                            </svg>
                            Guardar Asignaciones
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header bg-info text-white">
            <h3 class="mb-0">Asignaciones Existentes</h3>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Coordinador</th>
                            <th>Postgrado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($asignaciones->num_rows > 0): ?>
                            <?php while ($asignacion = $asignaciones->fetch_assoc()): ?>
                                <tr>
                                    <td><?= htmlspecialchars($asignacion['coordinador']) ?></td>
                                    <td><?= htmlspecialchars($asignacion['postgrado']) ?></td>
                                    <td>
                                        <a href='editar_asignacion.php?id=<?= $asignacion['id'] ?>' class="btn btn-sm btn-primary">Editar</a>
                                        <a href='asignar_postgrados.php?eliminar=<?= $asignacion['id'] ?>' class="btn btn-sm btn-danger" onclick='return confirm("¿Está seguro de eliminar esta asignación?")'>Eliminar</a>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="3" class="text-center">No hay asignaciones registradas</td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    document.getElementById('coordinador_id').addEventListener('change', function() {
        const coordinadorId = this.value;
        
        if (!coordinadorId) {
            // Desmarcar todos los checkboxes si no hay coordinador seleccionado
            document.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
                checkbox.checked = false;
            });
            return;
        }
        
        // Obtener postgrados asignados a este coordinador
        fetch(`get_postgrados_asignados.php?coordinador_id=${coordinadorId}`)
            .then(response => response.json())
            .then(data => {
                // Desmarcar todos los checkboxes primero
                document.querySelectorAll('input[type="checkbox"]').forEach(checkbox => {
                    checkbox.checked = false;
                });
                
                // Marcar los checkboxes de los postgrados asignados
                data.forEach(postgrado => {
                    const checkbox = document.getElementById(`postgrado_${postgrado.id}`);
                    if (checkbox) checkbox.checked = true;
                });
            })
            .catch(error => {
                console.error('Error:', error);
            });
    });

    // Al cargar la página, si hay un coordinador seleccionado en el POST, cargar sus postgrados
    document.addEventListener('DOMContentLoaded', function() {
        const coordinadorId = document.getElementById('coordinador_id').value;
        if (coordinadorId) {
            document.getElementById('coordinador_id').dispatchEvent(new Event('change'));
        }
    });
</script>

<?php
include '../includes/footer.php';
$conn->close();
?>