<?php
session_start();

// Configurar encoding UTF-8
header('Content-Type: text/html; charset=utf-8');
mb_internal_encoding('UTF-8');

if (!isset($_SESSION['username']) || $_SESSION['rol'] != 'administrador') {
    header("Location: ../login.php");
    exit();
}

include '../includes/db.php';

// Configurar conexión para usar UTF-8
$conn->set_charset("utf8mb4");
$conn->query("SET NAMES 'utf8mb4'");
$conn->query("SET CHARACTER SET utf8mb4");
$conn->query("SET COLLATION_CONNECTION = 'utf8mb4_unicode_ci'");

// Obtener listados para filtros
$sql_postgrados = "SELECT * FROM postgrados";
$result_postgrados = $conn->query($sql_postgrados);

$sql_promociones = "SELECT * FROM promociones";
$result_promociones = $conn->query($sql_promociones);

$sql_escolaridades = "SELECT * FROM escolaridades";
$result_escolaridades = $conn->query($sql_escolaridades);

$sql_anios_lectivos = "SELECT * FROM años_lectivos";
$result_anios_lectivos = $conn->query($sql_anios_lectivos);

// Procesar el formulario para graduar estudiante
if (isset($_POST['graduar_estudiante'])) {
    $estudiante_id = intval($_POST['estudiante_id']);
    $postgrado_id = intval($_POST['postgrado_id']);
    $promocion_id = intval($_POST['promocion_id']);
    $fecha_graduacion = $_POST['fecha_graduacion'];
    
    // Validar datos
    if (empty($fecha_graduacion)) {
        $error = "La fecha de graduación es requerida";
    } else {
        // Actualizar la relación estudiante-postgrado
        $sql = "UPDATE estudiante_postgrado 
                SET estado = 'graduado', 
                    fecha_graduacion = ?,
                    promocion_id = ?
                WHERE estudiante_id = ? 
                AND postgrado_id = ?";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("siii", $fecha_graduacion, $promocion_id, $estudiante_id, $postgrado_id);
            
            if ($stmt->execute()) {
                $mensaje = "Estudiante graduado correctamente";
            } else {
                $error = "Error al graduar estudiante: " . $stmt->error;
            }
            $stmt->close();
        } else {
            $error = "Error en la consulta: " . $conn->error;
        }
    }
}

// Obtener parámetros de filtro
$postgrado_id = isset($_GET['postgrado_id']) ? intval($_GET['postgrado_id']) : 0;
$escolaridad_id = isset($_GET['escolaridad_id']) ? intval($_GET['escolaridad_id']) : 0;
$anio_lectivo_id = isset($_GET['anio_lectivo_id']) ? intval($_GET['anio_lectivo_id']) : 0;
$fecha_graduacion_filtro = isset($_GET['fecha_graduacion']) ? $_GET['fecha_graduacion'] : '';
$cedula_filtro = isset($_GET['cedula']) ? trim($_GET['cedula']) : '';

// Obtener estudiantes que han completado todas las denominaciones (sin graduar)
$estudiantes_completados = [];
if ($postgrado_id > 0) {
    $sql_estudiantes = "SELECT e.id, e.nombres, e.apellidos, e.cedula
                       FROM estudiantes e
                       JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                       WHERE ep.postgrado_id = ?
                       AND ep.estado = 'activo'
                       AND NOT EXISTS (
                           SELECT d.id
                           FROM denominaciones d
                           WHERE d.postgrado_id = ?
                           AND d.id NOT IN (
                               SELECT i.denominacion_id
                               FROM inscripciones i
                               WHERE i.estudiante_id = e.id
                               AND i.nota_definitiva IS NOT NULL
                           )
                       )";
    
    $stmt = $conn->prepare($sql_estudiantes);
    if ($stmt) {
        $stmt->bind_param("ii", $postgrado_id, $postgrado_id);
        if ($stmt->execute()) {
            $estudiantes_completados = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
        } else {
            $error = "Error al obtener estudiantes: " . $stmt->error;
        }
        $stmt->close();
    } else {
        $error = "Error en la consulta: " . $conn->error;
    }
}

// Obtener estudiantes ya graduados (con fecha de grado)
$sql_graduados = "SELECT e.id, e.nombres, e.apellidos, e.cedula, 
                 ep.fecha_graduacion, p.nombre AS postgrado_nombre, 
                 pr.nombre AS promocion_nombre,
                 ep.postgrado_id
                 FROM estudiantes e
                 JOIN estudiante_postgrado ep ON e.id = ep.estudiante_id
                 JOIN postgrados p ON ep.postgrado_id = p.id
                 LEFT JOIN promociones pr ON ep.promocion_id = pr.id
                 WHERE ep.estado = 'graduado'";
                 
$conditions = [];
$params = [];
$types = "";

if ($postgrado_id > 0) {
    $conditions[] = "ep.postgrado_id = ?";
    $params[] = $postgrado_id;
    $types .= "i";
}

if ($escolaridad_id > 0) {
    // Necesitamos unir la tabla inscripciones para este filtro
    $sql_graduados .= " JOIN inscripciones i ON e.id = i.estudiante_id";
    $conditions[] = "i.escolaridad_id = ?";
    $params[] = $escolaridad_id;
    $types .= "i";
}

if ($anio_lectivo_id > 0) {
    // Necesitamos unir la tabla inscripciones para este filtro
    if (strpos($sql_graduados, 'JOIN inscripciones i') === false) {
        $sql_graduados .= " JOIN inscripciones i ON e.id = i.estudiante_id";
    }
    $conditions[] = "i.año_lectivo_id = ?";
    $params[] = $anio_lectivo_id;
    $types .= "i";
}

if (!empty($fecha_graduacion_filtro)) {
    $conditions[] = "ep.fecha_graduacion = ?";
    $params[] = $fecha_graduacion_filtro;
    $types .= "s";
}

if (!empty($cedula_filtro)) {
    $conditions[] = "e.cedula LIKE ?";
    $params[] = "%$cedula_filtro%";
    $types .= "s";
}

if (!empty($conditions)) {
    $sql_graduados .= " AND " . implode(" AND ", $conditions);
}

$sql_graduados .= " GROUP BY e.id ORDER BY ep.fecha_graduacion DESC";

$stmt_graduados = $conn->prepare($sql_graduados);
if ($stmt_graduados) {
    if (!empty($params)) {
        $stmt_graduados->bind_param($types, ...$params);
    }
    if ($stmt_graduados->execute()) {
        $graduados = $stmt_graduados->get_result()->fetch_all(MYSQLI_ASSOC);
    } else {
        $error = "Error al obtener graduados: " . $stmt_graduados->error;
    }
    $stmt_graduados->close();
} else {
    $error = "Error en la consulta de graduados: " . $conn->error;
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Gestión de Graduación</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: left;
        }
        th {
            background-color: #f4f4f4;
        }
        .form-container {
            margin-bottom: 20px;
            padding: 15px;
            background-color: #f9f9f9;
            border-radius: 5px;
        }
        select, input, button {
            padding: 8px;
            margin: 5px;
        }
        .section {
            margin-bottom: 30px;
        }
        .btn-editar {
            background-color: #4CAF50;
            color: white;
            padding: 5px 10px;
            text-decoration: none;
            border-radius: 4px;
        }
        .error {
            color: red;
            margin: 10px 0;
        }
        .success {
            color: green;
            margin: 10px 0;
        }
        .filtros {
            display: flex;
            flex-wrap: wrap;
            gap: 15px;
            margin-bottom: 15px;
        }
        .filtro-item {
            min-width: 200px;
        }
    </style>
</head>
<body>
    <h1>Gestión de Graduación</h1>
    <a href="../admin/index.php">← Volver al Menú</a>

    <?php if (isset($error)): ?>
        <div class="error"><?= $error ?></div>
    <?php endif; ?>
    
    <?php if (isset($mensaje)): ?>
        <div class="success"><?= $mensaje ?></div>
    <?php endif; ?>

    <!-- Filtros para ambos listados -->
    <div class="form-container">
        <h2>Filtros</h2>
        <form method="get">
            <div class="filtros">
                <div class="filtro-item">
                    <label><strong>Postgrado:</strong>
                        <select name="postgrado_id" onchange="this.form.submit()">
                            <option value="">Todos</option>
                            <?php 
                            $result_postgrados->data_seek(0);
                            while($p = $result_postgrados->fetch_assoc()): ?>
                                <option value="<?= $p['id'] ?>" <?= $postgrado_id == $p['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($p['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </label>
                </div>
                
                <div class="filtro-item">
                    <label><strong>Escolaridad:</strong>
                        <select name="escolaridad_id">
                            <option value="">Todas</option>
                            <?php 
                            $result_escolaridades->data_seek(0);
                            while($e = $result_escolaridades->fetch_assoc()): ?>
                                <option value="<?= $e['id'] ?>" <?= $escolaridad_id == $e['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($e['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </label>
                </div>
                
                <div class="filtro-item">
                    <label><strong>Año Lectivo:</strong>
                        <select name="anio_lectivo_id">
                            <option value="">Todos</option>
                            <?php 
                            $result_anios_lectivos->data_seek(0);
                            while($a = $result_anios_lectivos->fetch_assoc()): ?>
                                <option value="<?= $a['id'] ?>" <?= $anio_lectivo_id == $a['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($a['nombre']) ?>
                                </option>
                            <?php endwhile; ?>
                        </select>
                    </label>
                </div>
                
                <div class="filtro-item">
                    <label><strong>Fecha de Graduación:</strong>
                        <input type="date" name="fecha_graduacion" value="<?= htmlspecialchars($fecha_graduacion_filtro) ?>">
                    </label>
                </div>
                
                <div class="filtro-item">
                    <label><strong>Cédula:</strong>
                        <input type="text" name="cedula" placeholder="Buscar por cédula" value="<?= htmlspecialchars($cedula_filtro) ?>">
                    </label>
                </div>
            </div>
            
            <button type="submit">Aplicar Filtros</button>
            <a href="gestion_graduacion.php"><button type="button">Limpiar Filtros</button></a>
        </form>
    </div>

    <!-- Estudiantes que han completado todas las denominaciones (sin graduar) -->
    <div class="section">
        <h2>Estudiantes que han completado todas las denominaciones</h2>
        
        <?php if ($postgrado_id > 0): ?>
            <?php if (!empty($estudiantes_completados)): ?>
                <table>
                    <tr>
                        <th>ID</th>
                        <th>Nombre</th>
                        <th>Apellidos</th>
                        <th>Cédula</th>
                        <th>Acciones</th>
                    </tr>
                    <?php foreach ($estudiantes_completados as $estudiante): ?>
                        <tr>
                            <td><?= $estudiante['id'] ?></td>
                            <td><?= htmlspecialchars($estudiante['nombres']) ?></td>
                            <td><?= htmlspecialchars($estudiante['apellidos']) ?></td>
                            <td><?= htmlspecialchars($estudiante['cedula']) ?></td>
                            <td>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="estudiante_id" value="<?= $estudiante['id'] ?>">
                                    <input type="hidden" name="postgrado_id" value="<?= $postgrado_id ?>">
                                    <label><strong>Promoción:</strong>
                                        <select name="promocion_id" required>
                                            <option value="">-- Seleccione --</option>
                                            <?php 
                                            $result_promociones->data_seek(0);
                                            while($prom = $result_promociones->fetch_assoc()): ?>
                                                <option value="<?= $prom['id'] ?>"><?= htmlspecialchars($prom['nombre']) ?></option>
                                            <?php endwhile; ?>
                                        </select>
                                    </label>
                                    <label><strong>Fecha de Graduación:</strong>
                                        <input type="date" name="fecha_graduacion" required>
                                    </label>
                                    <button type="submit" name="graduar_estudiante">Graduar Estudiante</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            <?php else: ?>
                <p>No hay estudiantes que hayan completado todas las denominaciones de este postgrado.</p>
            <?php endif; ?>
        <?php else: ?>
            <p>Seleccione un postgrado para ver los estudiantes que han completado todas las denominaciones.</p>
        <?php endif; ?>
    </div>

    <!-- Estudiantes ya graduados -->
    <div class="section">
        <h2>Estudiantes Graduados</h2>
        
        <?php if (!empty($graduados)): ?>
            <table>
                <tr>
                    <th>ID</th>
                    <th>Nombre</th>
                    <th>Apellidos</th>
                    <th>Cédula</th>
                    <th>Postgrado</th>
                    <th>Promoción</th>
                    <th>Fecha de Graduación</th>
                    <th>Acciones</th>
                </tr>
                <?php foreach ($graduados as $graduado): ?>
                    <tr>
                        <td><?= $graduado['id'] ?></td>
                        <td><?= htmlspecialchars($graduado['nombres']) ?></td>
                        <td><?= htmlspecialchars($graduado['apellidos']) ?></td>
                        <td><?= htmlspecialchars($graduado['cedula']) ?></td>
                        <td><?= htmlspecialchars($graduado['postgrado_nombre']) ?></td>
                        <td><?= htmlspecialchars($graduado['promocion_nombre'] ?? 'Sin promoción') ?></td>
                        <td><?= htmlspecialchars($graduado['fecha_graduacion']) ?></td>
                        <td>
                            <form method="post" style="display:inline;">
                                <input type="hidden" name="estudiante_id" value="<?= $graduado['id'] ?>">
                                <input type="hidden" name="postgrado_id" value="<?= $graduado['postgrado_id'] ?>">
                                <label><strong>Promoción:</strong>
                                    <select name="promocion_id" required>
                                        <option value="">-- Seleccione --</option>
                                        <?php 
                                        $result_promociones->data_seek(0);
                                        while($prom = $result_promociones->fetch_assoc()): ?>
                                            <option value="<?= $prom['id'] ?>" <?= ($graduado['promocion_nombre'] ?? '') == $prom['nombre'] ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($prom['nombre']) ?>
                                            </option>
                                        <?php endwhile; ?>
                                    </select>
                                </label>
                                <label><strong>Fecha:</strong>
                                    <input type="date" name="fecha_graduacion" value="<?= htmlspecialchars($graduado['fecha_graduacion']) ?>" required>
                                </label>
                                <button type="submit" name="graduar_estudiante">Actualizar Graduación</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        <?php else: ?>
            <p>No se encontraron estudiantes graduados con los filtros aplicados.</p>
        <?php endif; ?>
    </div>
</body>
</html>